<?php
/**
 * Assignment API
 *
 * @author Nhamdv <daonham95@gmail.com>
 */
if ( class_exists( 'LP_REST_Jwt_Posts_Controller' ) ) {
	class LP_Jwt_Assignment_V1_Controller extends LP_REST_Jwt_Posts_Controller {
		protected $namespace = 'learnpress/v1';

		protected $rest_base = 'assignments';

		protected $post_type = LP_ASSIGNMENT_CPT;

		protected $hierarchical = true;

		public function register_routes() {
			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base,
				array(
					array(
						'methods'             => WP_REST_Server::READABLE,
						'callback'            => array( $this, 'get_items' ),
						'permission_callback' => array( $this, 'get_items_permissions_check' ),
						'args'                => $this->get_collection_params(),
					),
					'schema' => array( $this, 'get_public_item_schema' ),
				)
			);

			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base . '/(?P<id>[\d]+)',
				array(
					'args'   => array(
						'id' => array(
							'description' => esc_html__( 'Unique identifier for the resource.', 'learnpress' ),
							'type'        => 'integer',
						),
					),
					array(
						'methods'             => WP_REST_Server::READABLE,
						'callback'            => array( $this, 'get_item' ),
						'permission_callback' => array( $this, 'get_item_permissions_check' ),
						'args'                => array(
							'context' => $this->get_context_param(
								array(
									'default' => 'view',
								)
							),
						),
					),
					'schema' => array( $this, 'get_public_item_schema' ),
				)
			);

			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base . '/start',
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'start_assignment' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'id' => array(
							'description'       => esc_html__( 'Unique identifier for the resource.', 'learnpress' ),
							'type'              => 'integer',
							'required'          => true,
							'sanitize_callback' => 'absint',
						),
					),
				)
			);

			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base . '/retake',
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'retake_assignment' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'id' => array(
							'description'       => esc_html__( 'Unique identifier for the resource.', 'learnpress' ),
							'type'              => 'integer',
							'required'          => true,
							'sanitize_callback' => 'absint',
						),
					),
				)
			);

			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base . '/submit',
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'submit_assignment' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'action' => array(
							'description'       => esc_html__( 'Send or Save', 'learnpress' ),
							'type'              => 'string',
							'default'           => 'send',
							'sanitize_callback' => 'sanitize_text_field',
						),
						'id'     => array(
							'description'       => esc_html__( 'Unique identifier for the resource.', 'learnpress' ),
							'type'              => 'integer',
							'required'          => true,
							'sanitize_callback' => 'absint',
						),
						'note'   => array(
							'description'       => esc_html__( 'User note.', 'learnpress' ),
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => 'sanitize_text_field',
						),
						'file'   => array(
							'description' => esc_html__( 'File.', 'learnpress' ),
							'type'        => 'array',
						),
					),
				)
			);

			register_rest_route(
				$this->namespace,
				'/' . $this->rest_base . '/delete-submit-file',
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'delete_submit_file' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'fileId' => array(
							'description'       => esc_html__( 'File ID', 'learnpress' ),
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => 'sanitize_text_field',
						),
						'id'     => array(
							'description'       => esc_html__( 'Unique identifier for the resource.', 'learnpress' ),
							'type'              => 'integer',
							'required'          => true,
							'sanitize_callback' => 'absint',
						),
					),
				)
			);
		}

		public function retake_assignment( $request ) {
			$response = array(
				'data'    => array(
					'status' => 400,
				),
				'message' => esc_html__( 'There was an error starting the assignment.', 'learnpress-assignment' ),
			);

			try {
				$id        = $request->get_param( 'id' );
				$user      = learn_press_get_current_user();
				$course_id = $this->get_course_by_item_id( $id );

				if ( ! $course_id ) {
					throw new Exception( esc_html__( 'This Assignment need assign in a course', 'learnpress-assignment' ) );
				}

				if ( ! $user ) {
					throw new Exception( esc_html__( 'Username is not available!', 'learnpress-assignment' ) );
				}

				if ( ! $user->has_item_status( array( 'completed', 'evaluated' ), $id, $course_id ) ) {
					throw new Exception( esc_html__( 'You cannot retake this assignment.', 'learnpress-assignment' ) );
				}

				$assignment = new LP_Assignment( $id );

				$retake_count = $assignment->get_retake_count();

				$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

				$retaken = learn_press_get_user_item_meta( $user_item_id, '_lp_assignment_retaken', true );
				$retaken = ! empty( $retaken ) ? absint( $retaken ) : 0;

				if ( absint( $retake_count ) <= $retaken ) {
					throw new Exception( esc_html__( 'You cannot retake this assignment.', 'learnpress-assignment' ) );
				}

				$retake = learn_press_assignment_start( $user, $id, $course_id, 'retake', true );

				if ( is_wp_error( $retake ) ) {
					throw new Exception( $retake->get_error_message() );
				} else {
					$response = array(
						'data'    => array(
							'status' => 200,
						),
						'message' => esc_html__( 'Your Assignment has been started successfully.', 'learnpress-assignment' ),
					);
				}
			} catch ( \Throwable $th ) {
				$response['message'] = $th->getMessage();
			}

			return rest_ensure_response( $response );
		}

		public function delete_submit_file( $request ) {
			global $wp_filesystem;

			require_once ABSPATH . '/wp-admin/includes/file.php';
			WP_Filesystem();

			$response = array(
				'data'    => array(
					'status' => 400,
				),
				'message' => esc_html__( 'There was an error delete file in assignment.', 'learnpress-assignment' ),
			);

			try {
				$file_id       = $request->get_param( 'fileId' );
				$id            = $request->get_param( 'id' );
				$user          = learn_press_get_current_user();
				$assignment_db = LP_Assigment_DB::getInstance();
				$course_id     = $this->get_course_by_item_id( $id );

				if ( ! $course_id ) {
					throw new Exception( esc_html__( 'This Assignment need assign in a course', 'learnpress-assignment' ) );
				}

				if ( ! $user ) {
					throw new Exception( esc_html__( 'Username is not available!', 'learnpress-assignment' ) );
				}

				$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

				if ( ! $user->has_item_status( array( 'started', 'doing' ), $id, $course_id ) ) {
					throw new Exception( esc_html__( 'You cannot delete this file!', 'learnpress-assignment' ) );
				}

				$uploaded_files = learn_press_assignment_get_uploaded_files( $user_item_id );
				$file           = $uploaded_files[ $file_id ]->file ?? '';

				unset( $uploaded_files[ $file_id ] );

				if ( $wp_filesystem->delete( ABSPATH . $file, false, 'f' ) ) {
					$assignment_db->update_extra_value( $user_item_id, $assignment_db::$answer_upload_key, json_encode( $uploaded_files ) );
				} else {
					throw new Exception( esc_html__( 'Remove file failed, maybe there is issue with the permission.', 'learnpress-assignment' ) );
				}

				$response['data']['status'] = 200;
				$response['message']        = esc_html__( 'Remove file successfully', 'learnpress-assignment' );
			} catch ( \Throwable $th ) {
				$response['data']['status'] = 400;
				$response['message']        = $th->getMessage();
			}

			return rest_ensure_response( $response );
		}

		public function submit_assignment( $request ) {
			$response = array(
				'data'    => array(
					'status' => 400,
				),
				'message' => esc_html__( 'There was an error starting the assignment.', 'learnpress-assignment' ),
			);

			try {
				$action    = $request->get_param( 'action' );
				$id        = $request->get_param( 'id' );
				$note      = $request->get_param( 'note' );
				$files     = $request->get_file_params();
				$user      = learn_press_get_current_user();
				$course_id = $this->get_course_by_item_id( $id );

				if ( ! $course_id ) {
					throw new Exception( esc_html__( 'This Assignment need assign in a course', 'learnpress-assignment' ) );
				}

				if ( ! $user ) {
					throw new Exception( esc_html__( 'Username is not available!', 'learnpress-assignment' ) );
				}

				if ( $user->has_item_status( array( 'completed' ), $id, $course_id ) ) {
					throw new Exception( esc_html__( 'You is already send Answer, please wait the evaluated result!', 'learnpress-assignment' ) );
				}

				if ( ! $user->has_item_status( array( 'started', 'doing' ), $id, $course_id ) ) {
					throw new Exception( esc_html__( 'Please start Assignment to continue!', 'learnpress-assignment' ) );
				}

				$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

				$assignment_db = LP_Assigment_DB::getInstance();

				$assignment_db->update_extra_value( $user_item_id, LP_Assigment_DB::$answer_note_key, $note );

				if ( ! empty( $files ) ) {
					$files = $files['file'];

					if ( ! empty( $files['name'][0] ) ) {
						$allow_file_amount  = get_post_meta( $id, '_lp_upload_files', true );
						$uploaded_files     = learn_press_assignment_get_uploaded_files( $user_item_id );
						$total_files        = $uploaded_files ? count( $uploaded_files ) : 0;
						$file_extension     = get_post_meta( $id, '_lp_file_extension', true );
						$file_extension     = $file_extension ? preg_replace( '#\s#', '', $file_extension ) : '*';
						$file_extension_arr = explode( ',', $file_extension );
						$max_upload_size    = get_post_meta( $id, '_lp_upload_file_limit', true );

						add_filter(
							'upload_dir',
							function( $dir ) use ( $id, $user ) {
								$more_path = '/assignments';

								if ( isset( $id ) && $id ) {
									$more_path .= '/' . $id;
								}

								if ( isset( $user ) && $user->get_id() ) {
									$more_path .= '/' . $user->get_id();
								}

								$dir['path']   = $dir['basedir'] . $more_path;
								$dir['url']    = $dir['baseurl'] . $more_path;
								$dir['subdir'] = $more_path;

								return $dir;
							}
						);

						$file_uploaded = 0;

						foreach ( $files['name'] as $key => $value ) {
							if ( $total_files >= $allow_file_amount ) {
								throw new Exception( esc_html__( 'Your uploaded files reach the maximum amount!', 'learnpress-assignment' ) );
							}

							$file = array(
								'name'     => $files['name'][ $key ],
								'type'     => $files['type'][ $key ],
								'tmp_name' => $files['tmp_name'][ $key ],
								'error'    => $files['error'][ $key ],
								'size'     => $files['size'][ $key ],
							);

							if ( $file['size'] > $max_upload_size * 1024 * 1024 ) {
								$response['items'][ $key ]['message'] = sprintf( esc_html__( " The size of your %1\$s file is over %2\$d Mb(s).\n", 'learnpress-assignments' ), $file['name'], $max_upload_size );
								continue;
							}

							if ( ! in_array( '*', $file_extension_arr ) ) {
								$ext = wp_check_filetype( $file['name'] )['ext'];

								if ( $ext && ! in_array( strtolower( $ext ), $file_extension_arr ) ) {
									$response['items'][ $key ]['message'] = sprintf( esc_html__( '%s is not allowed!', 'learnpress-assignments' ), $file['name'] );
									continue;
								}
							}

							// Include filesystem functions to get access to wp_handle_upload().
							require_once ABSPATH . 'wp-admin/includes/file.php';

							$movefile = wp_handle_upload( $file, array( 'test_form' => false ) );

							if ( $movefile && ! isset( $movefile['error'] ) ) {
								$movefile['filename']                       = $files['name'][ $key ];
								$movefile['file']                           = str_replace( ABSPATH, '', $movefile['file'] );
								$movefile['url']                            = str_replace( get_site_url(), '', $movefile['url'] );
								$movefile['saved_time']                     = current_time( 'Y-m-d H:i:s' );
								$movefile['size']                           = $file['size'];
								$uploaded_files[ md5( $movefile['file'] ) ] = $movefile;
								$total_files ++;
								$file_uploaded ++;
							} else {
								$response['items'][ $key ]['message'] = sprintf( esc_html__( ' Please check file %s!', 'learnpress-assignments' ), $files['name'][ $key ] );
							}
						}

						remove_filter(
							'upload_dir',
							function( $dir ) use ( $id, $user ) {
								$more_path = '/assignments';

								if ( isset( $id ) && $id ) {
									$more_path .= '/' . $id;
								}

								if ( isset( $user ) && $user->get_id() ) {
									$more_path .= '/' . $user->get_id();
								}

								$dir['path']   = $dir['basedir'] . $more_path;
								$dir['url']    = $dir['baseurl'] . $more_path;
								$dir['subdir'] = $more_path;

								return $dir;
							}
						);

						$assignment_db->update_extra_value( $user_item_id, $assignment_db::$answer_upload_key, json_encode( $uploaded_files ) );

						if ( $file_uploaded !== count( $files['name'] ) ) {
							throw new Exception( esc_html__( 'Some file uploaded error!', 'learnpress-assignment' ) );
						}
					}
				}

				learn_press_update_assignment_item( $id, $course_id, $user, 'doing', $user_item_id );

				$response = array(
					'data'    => array(
						'status' => 200,
					),
					'message' => esc_html__( 'The progress was saved! Your file(s) were uploaded successfully!', 'learnpress-assignments' ),
				);

				if ( $action === 'send' ) {
					$evaluate_author = learn_press_get_user_item_meta( $user_item_id, '_lp_assignment_evaluate_author', true );

					if ( ! $evaluate_author ) {
						learn_press_update_user_item_meta( $user_item_id, '_lp_assignment_evaluate_author', 0 );
					}

					learn_press_update_assignment_item( $id, $course_id, $user, 'completed', $user_item_id );

					$response['message'] = esc_html__( 'What you did was sent to the instructors, please wait the evaluated result!', 'learnpress-assignments' );

					do_action( 'learn-press/assignment/student-submitted', $user->get_id(), $id );
				}
			} catch ( \Throwable $th ) {
				$response['data']['status'] = 400;
				$response['message']        = $th->getMessage();
			}

			return rest_ensure_response( $response );
		}

		public function start_assignment( $request ) {
			$response = array(
				'data'    => array(
					'status' => 400,
				),
				'message' => esc_html__( 'There was an error starting the assignment.', 'learnpress-assignment' ),
			);

			try {
				$id        = $request->get_param( 'id' );
				$user      = learn_press_get_current_user();
				$course_id = $this->get_course_by_item_id( $id );

				if ( ! $course_id ) {
					throw new Exception( esc_html__( 'This Assignment need assign in a course', 'learnpress-assignment' ) );
				}

				if ( ! $user ) {
					throw new Exception( esc_html__( 'Username is not available!', 'learnpress-assignment' ) );
				}

				if ( ! $user->has_course_status( $course_id, array( 'enrolled' ) ) || $user->has_item_status( array( 'started', 'doing', 'completed', 'evaluated' ), $id, $course_id ) ) {
					throw new Exception( esc_html__( 'You cannot start this Assignment', 'learnpress-assignment' ) );
				}

				$start = learn_press_assignment_start( $user, $id, $course_id, 'start', true );

				if ( is_wp_error( $start ) ) {
					throw new Exception( $start->get_error_message() );
				} else {
					$response = array(
						'data'    => array(
							'status' => 200,
						),
						'message' => esc_html__( 'Your Assignment has been started successfully.', 'learnpress-assignment' ),
					);
				}
			} catch ( \Throwable $th ) {
				$response['message'] = $th->getMessage();
			}

			return rest_ensure_response( $response );
		}

		public function prepare_object_for_response( $object, $request ) {
			$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
			$data    = $this->get_assignment_data( $object, $context, $request );

			$response = rest_ensure_response( $data );

			return apply_filters( "lp_jwt_rest_prepare_{$this->post_type}_object", $response, $object, $request );
		}

		protected function get_assignment_data( $object, $context = 'view' ) {
			$request = func_num_args() >= 2 ? func_get_arg( 2 ) : new WP_REST_Request( '', '', array( 'context' => $context ) );
			$fields  = $this->get_fields_for_response( $request );

			$id   = ! empty( $object->ID ) ? $object->ID : $object->get_id();
			$post = get_post( $id );
			$data = array();

			$assigned = $this->get_assigned( $id );

			$course_id = 0;

			if ( ! empty( $assigned ) && method_exists( $object, 'set_course' ) ) {
				$course_id = $assigned['course']['id'];
				$object->set_course( $course_id );
			}

			foreach ( $fields as $field ) {
				switch ( $field ) {
					case 'id':
						$data['id'] = $id;
						break;
					case 'name':
						$data['name'] = $post->post_title;
						break;
					case 'slug':
						$data['slug'] = $post->post_name;
						break;
					case 'permalink':
						$data['permalink'] = $object->get_permalink();
						break;
					case 'date_created':
						$data['date_created'] = lp_jwt_prepare_date_response( $post->post_date_gmt, $post->post_date );
						break;
					case 'date_created_gmt':
						$data['date_created_gmt'] = lp_jwt_prepare_date_response( $post->post_date_gmt );
						break;
					case 'date_modified':
						$data['date_modified'] = lp_jwt_prepare_date_response( $post->post_modified_gmt, $post->post_modified );
						break;
					case 'date_modified_gmt':
						$data['date_modified_gmt'] = lp_jwt_prepare_date_response( $post->post_modified_gmt );
						break;
					case 'status':
						$data['status'] = $post->post_status;
						break;
					case 'content':
						$data['content'] = 'view' === $context ? apply_filters( 'the_content', $post->post_content ) : $post->post_content;
						break;
					case 'excerpt':
						$data['excerpt'] = $post->post_excerpt;
						break;
					case 'assigned':
						$data['assigned'] = $assigned;
						break;
					case 'retake_count':
						$data['retake_count'] = absint( $object->get_retake_count() );
						break;
					case 'retaken':
						$data['retaken'] = $this->get_retaken( $id, $course_id );
						break;
					case 'duration':
						$data['duration'] = $this->get_duration( $object );
						break;
					case 'introdution':
						$data['introdution'] = $object->get_introduction();
						break;
					case 'passing_grade':
						$data['passing_grade'] = $object->get_passing_grade();
						break;
					case 'allow_file_type':
						$data['allow_file_type'] = $object->get_file_extension();
						break;
					case 'files_amount':
						$data['files_amount'] = absint( $object->get_files_amount() );
						break;
					case 'attachment':
						$data['attachment'] = $this->get_attachment( $object, $course_id );
						break;
					case 'results':
						$data['results'] = $this->get_results( $id, $course_id );
						break;
					case 'assignment_answer':
						$data['assignment_answer'] = $this->get_assignment_answer( $id, $course_id );
						break;
					case 'evaluation':
						$data['evaluation'] = $this->get_evaluation( $id, $course_id );
						break;
				}
			}

			return $data;
		}

		/**
		 * @param int $id
		 * @param int $course_id
		 *
		 * @return array|void
		 */
		public function get_evaluation( int $id, int $course_id ): array {
			$output        = array();
			$user          = learn_press_get_current_user();
			$assignment_db = LP_Assigment_DB::getInstance();

			if ( empty( $user ) || empty( $course_id ) || empty( $id ) ) {
				return $output;
			}

			if ( ! $user->has_item_status( array( 'evaluated' ), $id, $course_id ) ) {
				return $output;
			}

			$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

			$reference_files = learn_press_get_user_item_meta( $user_item_id, '_lp_assignment_evaluate_upload', true );
			// $instructor_note = learn_press_get_user_item_meta( $user_item_id, '_lp_assignment_instructor_note', true );
			$instructor_note = $assignment_db->get_extra_value( $this->user_item_id, $assignment_db::$instructor_note_key );
			if ( empty( $note ) ) { // get value old from column meta_value
				$instructor_note = learn_press_get_user_item_meta( $this->user_item_id, $assignment_db::$instructor_note_key, true );
			}

			$result = learn_press_assignment_get_result( $id, $user->get_id(), $course_id );

			$output['mark']       = $result['mark'] ?? '';
			$output['user_mark']  = $result['user_mark'] ?? '';
			$output['graduation'] = $result['grade'] ?? '';
			$output['result']     = $result['result'] ?? '';

			if ( ! empty( $reference_files ) ) {
				foreach ( $reference_files as $attachment ) {
					$output['reference_files'][] = array(
						'id'   => $attachment,
						'url'  => wp_get_attachment_url( $attachment ),
						'name' => wp_basename( wp_get_attachment_url( $attachment ) ),
					);
				}
			} else {
				$output['reference_files'] = array();
			}

			$output['instructor_note'] = ! empty( $instructor_note ) ? $instructor_note : '';

			return $output;
		}

		public function get_assignment_answer( $id, $course_id ) {
			$output = array();
			$user   = learn_press_get_current_user();

			if ( empty( $user ) || empty( $course_id ) || empty( $id ) ) {
				return $output;
			}

			$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

			if ( ! $user->has_item_status( array( 'started', 'doing', 'completed', 'evaluated' ), $id, $course_id ) ) {
				return $output;
			}

			$assignment_db = LP_Assigment_DB::getInstance();
			$content       = $assignment_db->get_extra_value( $user_item_id, $assignment_db::$answer_note_key );

			$output['note'] = ! empty( $content ) ? $content : '';

			$uploaded_files = learn_press_assignment_get_uploaded_files( $user_item_id );

			$output['file'] = $uploaded_files;

			return $output;
		}

		public function get_results( $id, $course_id ) {
			$user = learn_press_get_current_user();

			$output = array();

			if ( empty( $user ) || empty( $course_id ) || empty( $id ) ) {
				return $output;
			}

			$user_course = $user->get_course_data( $course_id );
			$user_item   = $user_course ? $user_course->get_item( $id ) : false;

			if ( $user_item ) {
				$start_time      = $user_item->get_start_time();
				$expiration_time = $user_item->get_expiration_time();
				$end_time        = $user_item->get_end_time();

				$output['status']          = $user_item->get_status();
				$output['start_time']      = $start_time ? lp_jwt_prepare_date_response( $start_time->toSql( false ) ) : '';
				$output['expiration_time'] = $expiration_time ? lp_jwt_prepare_date_response( $expiration_time->toSql( false ) ) : '';
				$output['end_time']        = $end_time ? lp_jwt_prepare_date_response( $end_time->toSql( false ) ) : '';
			}

			return $output;
		}

		public function get_attachment( $object, $course_id ) {
			$user = learn_press_get_current_user();
			$id   = ! empty( $object->ID ) ? $object->ID : $object->get_id();

			$output = array();

			if ( empty( $user ) || empty( $course_id ) || empty( $id ) ) {
				return $output;
			}

			if ( $user->has_course_status( $course_id, array( 'finished' ) ) || $user->has_item_status( array( 'started', 'doing', 'completed', 'evaluated' ), $id, $course_id ) ) {
				$attachments = $object->get_attachments_assignment( $object );

				if ( ! empty( $attachments ) ) {
					foreach ( $attachments as $attachment ) {
						$output[] = array(
							'id'   => $attachment,
							'url'  => wp_get_attachment_url( $attachment ),
							'name' => wp_basename( wp_get_attachment_url( $attachment ) ),
						);
					}
				}
			}

			return $output;
		}

		public function get_duration( $item ) {
			$output = array();

			$format = array(
				'day'    => esc_html__( '%s days', 'learnpress-assignment' ),
				'hour'   => esc_html__( '%s hours', 'learnpress-assignment' ),
				'minute' => esc_html__( '%s mins', 'learnpress-assignment' ),
				'second' => esc_html__( '%s secs', 'learnpress-assignment' ),
			);

			$output['format'] = $item->get_duration() ? $item->get_duration()->to_timer( $format, true ) : '';
			$output['time']   = $item->get_duration() ? $item->get_duration()->get() : '';

			return $output;
		}

		public function get_retaken( $id, $course_id ) {
			$user = learn_press_get_current_user();

			if ( empty( $user ) || empty( $course_id ) || empty( $id ) ) {
				return 0;
			}

			$user_item_id = $this->get_user_item_id( $user, $id, $course_id );

			$redo_time = learn_press_get_user_item_meta( $user_item_id, '_lp_assignment_retaken', true );

			return $redo_time ? absint( $redo_time ) : 0;
		}

		public function get_user_item_id( $user, $id, $course_id ) {
			$user_item_id = 0;
			$course_data  = $user->get_course_data( $course_id );
			if ( $course_data ) {
				$user_item    = $course_data->get_item( $id );
				$user_item_id = $user_item ? $user_item->get_user_item_id() : 0;
			}

			return $user_item_id;
		}

		protected function get_object( $assignment = 0 ) {
			global $post;

			if ( false === $assignment && isset( $post, $post->ID ) && LP_ASSIGNMENT_CPT === get_post_type( $post->ID ) ) {
				$id = absint( $post->ID );
			} elseif ( is_numeric( $assignment ) ) {
				$id = $assignment;
			} elseif ( $assignment instanceof LP_Assignment ) {
				$id = $assignment->get_id();
			} elseif ( ! empty( $assignment->ID ) ) {
				$id = $assignment->ID;
			}

			return new LP_Assignment( $id );
		}

		/**
		 * Checks if a course can be read.
		 *
		 * Correctly handles courses with the inherit status.
		 *
		 * @author Nhamdv
		 *
		 * @return bool Whether the post can be read.
		 * */
		public function check_read_permission( $post_id ) {
			if ( empty( absint( $post_id ) ) ) {
				return false;
			}

			$post = get_post( $post_id );

			if ( ! $post ) {
				return false;
			}

			if ( lp_rest_check_post_permissions( $this->post_type, 'read', $post_id ) ) {
				return true;
			}

			$post_status_obj = get_post_status_object( $post->post_status );
			if ( ! $post_status_obj || ! $post_status_obj->public ) {
				return false;
			}

			$user_id = get_current_user_id();

			if ( ! $user_id ) {
				return false;
			}

			$user = learn_press_get_user( $user_id );

			// Get course ID by lesson ID assigned.
			$course_id = $this->get_course_by_item_id( $post_id );

			if ( empty( $course_id ) ) {
				return false;
			}

			$can_view_content_course = $user->can_view_content_course( $course_id );

			$can_view_item = $user->can_view_item( $post_id, $can_view_content_course );

			if ( ! $can_view_item->flag ) {
				return false;
			}

			// Can we read the parent if we're inheriting?
			if ( 'inherit' === $post->post_status && $post->post_parent > 0 ) {
				$parent = get_post( $post->post_parent );

				if ( $parent ) {
					return $this->check_read_permission( $parent );
				}
			}

			return true;
		}

		public function get_assigned( $id ) {
			$courses = learn_press_get_item_courses( $id );

			$output = array();

			if ( $courses ) {
				foreach ( $courses as $course ) {
					$output['course'] = array(
						'id'      => $course->ID,
						'title'   => $course->post_title,
						'slug'    => $course->post_name,
						'content' => $course->post_content,
						'author'  => $course->post_author,
					);
				}
			}

			return $output;
		}

		/**
		 * Get course ID by assignment ID assigned.
		 *
		 * @param [type] $item_id
		 * @return void
		 */
		protected function get_course_by_item_id( $item_id ) {
			static $output;

			global $wpdb;

			if ( empty( $item_id ) ) {
				return false;
			}

			if ( ! isset( $output ) ) {
				$output = $wpdb->get_var(
					$wpdb->prepare(
						"SELECT c.ID FROM {$wpdb->posts} c
					INNER JOIN {$wpdb->learnpress_sections} s ON c.ID = s.section_course_id
					INNER JOIN {$wpdb->learnpress_section_items} si ON si.section_id = s.section_id
					WHERE si.item_id = %d ORDER BY si.section_id DESC LIMIT 1
					",
						$item_id
					)
				);
			}

			if ( $output ) {
				return absint( $output );
			}

			return false;
		}

		public function get_item_schema() {
			$schema = array(
				'$schema'    => 'http://json-schema.org/draft-04/schema#',
				'title'      => $this->post_type,
				'type'       => 'object',
				'properties' => array(
					'id'                => array(
						'description' => __( 'Unique identifier for the resource.', 'learnpress' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'name'              => array(
						'description' => __( 'Assignment name.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'slug'              => array(
						'description' => __( 'Assignment slug.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'permalink'         => array(
						'description' => __( 'Assignment URL.', 'learnpress' ),
						'type'        => 'string',
						'format'      => 'uri',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'date_created'      => array(
						'description' => __( "The date the Course was created, in the site's timezone.", 'learnpress' ),
						'type'        => 'date-time',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'date_created_gmt'  => array(
						'description' => __( 'The date the Course was created, as GMT.', 'learnpress' ),
						'type'        => 'date-time',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'date_modified'     => array(
						'description' => __( "The date the Course was last modified, in the site's timezone.", 'learnpress' ),
						'type'        => 'date-time',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'date_modified_gmt' => array(
						'description' => __( 'The date the Course was last modified, as GMT.', 'learnpress' ),
						'type'        => 'date-time',
						'context'     => array( 'view', 'edit' ),
						'readonly'    => true,
					),
					'status'            => array(
						'description' => __( 'Assignment status (post status).', 'learnpress' ),
						'type'        => 'string',
						'default'     => 'publish',
						'enum'        => array_merge( array_keys( get_post_statuses() ), array( 'future' ) ),
						'context'     => array( 'view', 'edit' ),
					),
					'content'           => array(
						'description' => __( 'Content Assignment.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'excerpt'           => array(
						'description' => __( 'Retrieves the Assignment excerpt..', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'assigned'          => array(
						'description' => __( 'Assigned.', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view', 'edit' ),
						'items'       => array(
							'id'      => array(
								'description' => __( 'Item ID.', 'learnpress' ),
								'type'        => 'integer',
								'context'     => array( 'view', 'edit' ),
							),
							'title'   => array(
								'description' => __( 'Title.', 'learnpress' ),
								'type'        => 'string',
								'context'     => array( 'view', 'edit' ),
							),
							'slug'    => array(
								'description' => __( 'Item slug.', 'learnpress' ),
								'type'        => 'string',
								'context'     => array( 'view', 'edit' ),
							),
							'content' => array(
								'description' => __( 'Item Content.', 'learnpress' ),
								'type'        => 'string',
								'context'     => array( 'view', 'edit' ),
							),
							'author'  => array(
								'description' => __( 'Item Author.', 'learnpress' ),
								'type'        => 'integer',
								'context'     => array( 'view', 'edit' ),
							),
						),
					),
					'retake_count'      => array(
						'description' => __( 'Retake count.', 'learnpress' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
					'retaken'           => array(
						'description' => __( 'Retaken.', 'learnpress' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
					'duration'          => array(
						'description' => __( 'Duration.', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view', 'edit' ),
						'items'       => array(
							'format' => array(
								'description' => __( 'Format.', 'learnpress' ),
								'type'        => 'string',
								'context'     => array( 'view', 'edit' ),
							),
							'time'   => array(
								'description' => __( 'Time.', 'learnpress' ),
								'type'        => 'integer',
								'context'     => array( 'view', 'edit' ),
							),
						),
					),
					'introdution'       => array(
						'description' => __( 'Introdution.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'passing_grade'     => array(
						'description' => __( 'Passing grade.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'allow_file_type'   => array(
						'description' => __( 'Allow file type.', 'learnpress' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'files_amount'      => array(
						'description' => __( 'File amount can upload files.', 'learnpress' ),
						'type'        => 'integer',
						'context'     => array( 'view', 'edit' ),
					),
					'attachment'        => array(
						'description' => __( 'Attachment.', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view', 'edit' ),
					),
					'results'           => array(
						'description' => __( 'List of course user data.', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view' ),
						'items'       => array(
							'type'       => 'object',
							'properties' => array(
								'graduation'      => array(
									'description' => __( 'Graduation', 'learnpress' ),
									'type'        => 'string',
									'context'     => array( 'view' ),
								),
								'status'          => array(
									'description' => __( 'Status', 'learnpress' ),
									'type'        => 'string',
									'context'     => array( 'view' ),
									'readonly'    => true,
								),
								'start_time'      => array(
									'description' => __( 'Start time', 'learnpress' ),
									'type'        => 'string',
									'context'     => array( 'view' ),
									'readonly'    => true,
								),
								'end_time'        => array(
									'description' => __( 'End time', 'learnpress' ),
									'type'        => 'string',
									'context'     => array( 'view' ),
									'readonly'    => true,
								),
								'expiration_time' => array(
									'description' => __( 'Expiration time', 'learnpress' ),
									'type'        => 'string',
									'context'     => array( 'view' ),
									'readonly'    => true,
								),
							),
						),
					),
					'assignment_answer' => array(
						'description' => __( 'User answer Assignments', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view', 'edit' ),
					),
					'evaluation'        => array(
						'description' => __( 'Evaluation', 'learnpress' ),
						'type'        => 'array',
						'context'     => array( 'view', 'edit' ),
					),
				),
			);

			return $this->add_additional_fields_schema( $schema );
		}
	}
}
